import pygame
from pygame.locals import *

from game_consts import *

class GameObject(pygame.sprite.Sprite):
    def __init__(self, level, colnum, rownum, *groups):
        pygame.sprite.Sprite.__init__(self, *groups)

        self.level = level

        self.init_image() # Initialize self.image

        # Initialize arect from rownum, colnum
        self.arect = self.image.get_rect()
        self.arect.centerx = (colnum * TILE_WIDTH) + (TILE_WIDTH / 2)
        self.arect.bottom = (rownum + 1) * TILE_HEIGHT

    def update(self, *args):
        pass

class GameObjectNonAnimated(GameObject):
    def init_image(self):
        self.image = self.IMAGE.convert_alpha()

class GameObjectAnimated(GameObject):
    ANIMATION_DELAY = 100     # Change image every this many milliseconds
    
    def init_image(self):
        self.images_regular = [image.convert_alpha() for image in self.IMAGES]
        self.images_flipped = [pygame.transform.flip(image, True, False).convert_alpha() for image in self.IMAGES]

        self.animation_count = len(self.IMAGES)

        self.image = self.images_regular[0]

    def animate(self, flipped):
        if flipped:
            images = self.images_flipped
        else:
            images = self.images_regular

        animation_index = (pygame.time.get_ticks() / self.ANIMATION_DELAY) % self.animation_count
        self.image = images[animation_index]

class GameObjectAnimatedByKeyword(GameObject):
    ANIMATION_DELAY = 100     # Change image every this many milliseconds
    DEFAULT_ANIMATE_KEYWORD = "normal"

    def init_image(self):
        self.images_regular = {}
        self.images_flipped = {}
        self.animation_count = {}

        for k, v in self.IMAGES.iteritems():
            self.images_regular[k] = [image.convert_alpha() for image in v]
            self.images_flipped[k] = [pygame.transform.flip(image, True, False).convert_alpha() for image in v]

            self.animation_count[k] = len(v)

        self.last_flipped = False
        self.animate_keyword = self.DEFAULT_ANIMATE_KEYWORD
        self.image = self.images_regular[self.animate_keyword][0]

    def animate(self, flipped):
        if flipped:
            images = self.images_flipped[self.animate_keyword]
        else:
            images = self.images_regular[self.animate_keyword]

        self.last_flipped = flipped

        animation_index = (pygame.time.get_ticks() / self.ANIMATION_DELAY) % self.animation_count[self.animate_keyword]
        self.image = images[animation_index]

    def set_animate_keyword(self, animate_keyword):
        self.animate_keyword = animate_keyword
        self.animate(self.last_flipped)

class TextObject(GameObject):
    FONT_SIZE = 18 # before scale
    FONT_SCALE = 3
    TEXT_COLOR = (255, 255, 255)

    def init_image(self):
        self.font = pygame.font.Font(None, self.FONT_SIZE)
        self.update()
        
    def scale_font(self, image):
        x, y = image.get_size()
        x *= self.FONT_SCALE
        y *= self.FONT_SCALE
        x = int(round(x))
        y = int(round(y))
        return pygame.transform.scale(image, (x, y))

    def update(self, *args):
        self.update_text()
        self.image = self.font.render(self.text, False, self.TEXT_COLOR)
        self.image = self.scale_font(self.image)
        